-- name: Screenshot Mode
-- description: The screenshot mode from GeoGuessr! Useful for creators.\nMod by EmilyEmmi.

local ssHud = true
local cameraLock = false
local sMenuInputsPressed = 0
local sMenuInputsDown = 0
local frameAdvanceTimer = 0
local holdAdvanceTimer = 0

local LLS = gLakituState
local ssCamera = {
    pos = { x = 0, y = 0, z = 0 },
    pitch = 0,
    yaw = 0,
    roll = 0,
}

-- controls (uses menu controls too)
---@param m MarioState
function ss_controls(m)
    if m.playerIndex ~= 0 then return end

    if m.freeze < 3 then m.freeze = 3 end

    -- Disable controls for everything but the menu
    sMenuInputsPressed = m.controller.buttonDown & (m.controller.buttonDown ~ sMenuInputsDown)
    sMenuInputsDown = m.controller.buttonDown
    m.controller.buttonDown = 0
    m.controller.buttonPressed = 0
    m.controller.stickX = 0
    m.controller.stickY = 0

    LLS.posHSpeed = 0
    LLS.posVSpeed = 0
    LLS.focHSpeed = 0
    LLS.focVSpeed = 0
    vec3f_copy(LLS.pos, ssCamera.pos)
    vec3f_copy(LLS.curPos, ssCamera.pos)
    vec3f_copy(LLS.goalPos, ssCamera.pos)
    local focus = {
        x = ssCamera.pos.x - sins(ssCamera.yaw) * coss(ssCamera.pitch),
        y = ssCamera.pos.y + sins(ssCamera.pitch),
        z = ssCamera.pos.z - coss(ssCamera.yaw) * coss(ssCamera.pitch)
    }
    vec3f_copy(LLS.focus, focus)
    vec3f_copy(LLS.curFocus, focus)
    vec3f_copy(LLS.goalFocus, focus)
    LLS.roll = 0
    LLS.keyDanceRoll = ssCamera.roll
    m.area.camera.yaw = ssCamera.yaw

    if not cameraLock then
        local speed = 0.5
        local rSpeed = 0.6
        local vSpeed = 64
        local rollSpeed = 256
        local zoomSpeed = 128
        if sMenuInputsDown & B_BUTTON ~= 0 then
            speed = speed / 2
            rSpeed = rSpeed / 2
            vSpeed = vSpeed // 2
            rollSpeed = rollSpeed // 2
            zoomSpeed = zoomSpeed // 2
        end

        -- movement
        local stickX = m.controller.rawStickX
        local stickY = m.controller.rawStickY
        ssCamera.pos.x = ssCamera.pos.x -
            math.floor(speed * (sins(ssCamera.yaw) * stickY + sins(ssCamera.yaw - 16384) * stickX))
        ssCamera.pos.z = ssCamera.pos.z -
            math.floor(speed * (coss(ssCamera.yaw) * stickY + coss(ssCamera.yaw - 16384) * stickX))

        -- pitch/yaw
        local x_invert = (camera_config_is_free_cam_enabled() and camera_config_is_x_inverted() and -1) or 1
        local y_invert = (camera_config_is_free_cam_enabled() and camera_config_is_y_inverted() and -1) or 1
        local mouseX = 0
        local mouseY = 0
        if camera_config_is_mouse_look_enabled() then
            mouseX = x_invert * djui_hud_get_raw_mouse_x()
            mouseY = y_invert * djui_hud_get_raw_mouse_y()
        end
        local rStickX = x_invert * m.controller.extStickX
        local rStickY = y_invert * m.controller.extStickY
        if m.controller.extStickX == 0 then
            if (sMenuInputsDown & R_CBUTTONS) ~= 0 then
                rStickX = clamp(stickX - 128, -128, 128)
            end
            if (sMenuInputsDown & L_CBUTTONS) ~= 0 then
                rStickX = clamp(stickX + 128, -128, 128)
            end
        end
        if m.controller.extStickY == 0 then
            if (sMenuInputsDown & U_CBUTTONS) ~= 0 then
                rStickY = clamp(stickY + 128, -128, 128)
            end
            if (sMenuInputsDown & D_CBUTTONS) ~= 0 then
                rStickY = clamp(stickY - 128, -128, 128)
            end
        end
        ssCamera.yaw = limit_angle(ssCamera.yaw -
            (rSpeed * rStickX - mouseX) * (camera_config_get_x_sensitivity()))
        ssCamera.pitch = clamp(
            ssCamera.pitch - (rSpeed * rStickY + mouseY) * (camera_config_get_y_sensitivity()), -0x3E00,
            0x3E00)

        -- roll
        if sMenuInputsDown & U_JPAD ~= 0 then
            ssCamera.roll = 0
        end
        if sMenuInputsDown & L_JPAD ~= 0 then
            ssCamera.roll = limit_angle(ssCamera.roll - rollSpeed)
        end
        if sMenuInputsDown & R_JPAD ~= 0 then
            ssCamera.roll = limit_angle(ssCamera.roll + rollSpeed)
        end

        -- vertical movement
        if sMenuInputsDown & A_BUTTON ~= 0 then
            ssCamera.pos.y = ssCamera.pos.y + vSpeed
        end
        if sMenuInputsDown & Z_TRIG ~= 0 then
            ssCamera.pos.y = ssCamera.pos.y - vSpeed
        end

        -- zoom
        if sMenuInputsDown & R_TRIG ~= 0 then
            if sMenuInputsPressed & R_TRIG ~= 0 then
                play_sound(SOUND_MENU_CAMERA_ZOOM_IN, m.marioObj.header.gfx.cameraToObject)
            end
            ssCamera.pos.x = ssCamera.pos.x - math.floor(speed * coss(ssCamera.pitch) * sins(ssCamera.yaw) * zoomSpeed)
            ssCamera.pos.y = ssCamera.pos.y + math.floor(speed * sins(ssCamera.pitch) * zoomSpeed)
            ssCamera.pos.z = ssCamera.pos.z - math.floor(speed * coss(ssCamera.pitch) * coss(ssCamera.yaw) * zoomSpeed)
        end
        if sMenuInputsDown & L_TRIG ~= 0 then
            if sMenuInputsPressed & L_TRIG ~= 0 then
                play_sound(SOUND_MENU_CAMERA_ZOOM_OUT, m.marioObj.header.gfx.cameraToObject)
            end
            ssCamera.pos.x = ssCamera.pos.x + math.floor(speed * coss(ssCamera.pitch) * sins(ssCamera.yaw) * zoomSpeed)
            ssCamera.pos.y = ssCamera.pos.y - math.floor(speed * sins(ssCamera.pitch) * zoomSpeed)
            ssCamera.pos.z = ssCamera.pos.z + math.floor(speed * coss(ssCamera.pitch) * coss(ssCamera.yaw) * zoomSpeed)
        end
    end

    -- frame advance
    if sMenuInputsDown & D_JPAD ~= 0 then
        if frameAdvanceTimer == 0 or (holdAdvanceTimer >= 10 and frameAdvanceTimer <= 5) then
            frameAdvanceTimer = 10
        end
        holdAdvanceTimer = holdAdvanceTimer + 1
    else
        frameAdvanceTimer = 0
        holdAdvanceTimer = 0
    end

    if ((sMenuInputsPressed & X_BUTTON) ~= 0 or ((sMenuInputsDown & L_TRIG) ~= 0 and (sMenuInputsPressed & R_TRIG) ~= 0)) then
        play_sound(SOUND_MENU_CLICK_CHANGE_VIEW, m.marioObj.header.gfx.cameraToObject)
        cameraLock = not cameraLock
    elseif (sMenuInputsPressed & Y_BUTTON) ~= 0 then
        ssHud = not ssHud
    elseif (sMenuInputsPressed & START_BUTTON) ~= 0 then
        m.controller.buttonDown = m.controller.buttonDown | START_BUTTON
        screenshotMode = false
        LLS.roll = 0
        LLS.keyDanceRoll = 0
        disable_time_stop_including_mario()
        soft_reset_camera(m.area.camera)
    end
end

function screenshot_hud()
    if not ssHud then return end
    djui_hud_set_resolution(RESOLUTION_DJUI)
    djui_hud_set_font(FONT_MENU)
    djui_hud_set_color(255, 255, 255, 255)

    local screenWidth = djui_hud_get_screen_width()
    local screenHeight = djui_hud_get_screen_height()

    local scale = 1.5
    local text = "Screenshot Mode"
    local width = djui_hud_measure_text(text) * scale
    local x = (screenWidth - width) * 0.5
    local y = 20

    djui_hud_print_text(text, x, y, scale)

    local textLines = { "Control Stick - Move", "C Buttons/Right Stick: Rotate",
        "D-Pad: Tilt Left/Right (Up to reset)",
        "D-Pad Down: Advance frame",
        "R - Zoom In", "L - Zoom Out", "A - Up", "Z - Down", "Hold B - Slow", "X or L+R - Toggle lock",
        "Y - Toggle hud",
        "Start - Cancel" }
    scale = 0.5
    y = (screenHeight - #textLines * 60 * scale) * 0.5
    for i, line in ipairs(textLines) do
        x = 10
        djui_hud_print_text(line, x, y, scale)
        y = y + 60 * scale
    end
end

function enter_screenshot_mode()
    ssHud = true
    frameAdvanceTimer = 0
    holdAdvanceTimer = 0
    cameraLock = false
    screenshotMode = true
    vec3f_copy(ssCamera.pos, LLS.pos)
    local dist, pitch, yaw = vec3f_get_dist_and_angle_lua(LLS.pos, LLS.focus)
    ssCamera.yaw = limit_angle(yaw + 0x8000)
    ssCamera.pitch = pitch
    ssCamera.roll = LLS.roll
end

-- gets distance, pitch, and yaw between two points
function vec3f_get_dist_and_angle_lua(from, to)
    local x = to.x - from.x
    local y = to.y - from.y
    local z = to.z - from.z

    dist = math.sqrt(x * x + y * y + z * z)
    pitch = atan2s(math.sqrt(x * x + z * z), y)
    yaw = atan2s(z, x)
    return dist, pitch, yaw
end

-- from extended moveset
function limit_angle(a)
    return (a + 0x8000) % 0x10000 - 0x8000
end

function update()
    if screenshotMode then
        local m = gMarioStates[0]

        if frameAdvanceTimer == 10 then
            disable_time_stop() -- convention would tell you to use disable_time_stop_including_mario(), but that actually makes mario update for an extra frame for SOME reason
        else
            enable_time_stop_including_mario()
        end
        if frameAdvanceTimer > 0 then
            frameAdvanceTimer = frameAdvanceTimer - 1
        end

        if m.area.camera then
            if m.area.camera.cutscene ~= 0 then
                ss_controls(m)
            end
            m.area.camera.cutscene = 1
        end

        -- freeze particles too
        local o = obj_get_first(OBJ_LIST_UNIMPORTANT)
        while o do
            o.activeFlags = o.activeFlags & ~ACTIVE_FLAG_UNIMPORTANT
            o = obj_get_next(o)
        end
    end
end

hook_event(HOOK_UPDATE, update)

function before_mario_update(m)
    if m.playerIndex ~= 0 then return end

    if not (is_game_paused() or screenshotMode) and m.area.camera and m.area.camera.cutscene == 0 and m.controller.buttonDown & R_TRIG ~= 0 and m.controller.buttonPressed & L_TRIG ~= 0 then
        enter_screenshot_mode()
        sMenuInputsPressed = 0
        sMenuInputsDown = m.controller.buttonDown
        m.controller.buttonPressed = m.controller.buttonPressed & ~L_TRIG
    end
end

hook_event(HOOK_BEFORE_MARIO_UPDATE, before_mario_update)

function on_hud_render()
    if screenshotMode then
        toggle_hud(false)
        return screenshot_hud()
    else
        toggle_hud(true)
    end
end

hook_event(HOOK_ON_HUD_RENDER, on_hud_render)

-- compatibility with custom huds
local expectedHudState = true
function toggle_hud(show)
    if hud_is_hidden() == expectedHudState then return end
    if show then
        hud_show()
        expectedHudState = true
    else
        hud_hide()
        expectedHudState = false
    end
end

function ss_command()
    screenshotMode = not screenshotMode
    if screenshotMode then
        enter_screenshot_mode()
    else
        LLS.roll = 0
        LLS.keyDanceRoll = 0
        disable_time_stop_including_mario()
        soft_reset_camera(gMarioStates[0].area.camera)
    end
    return true
end

hook_chat_command("screenshot", "- Enter screenshot mode (or use L+R)", ss_command)

-- don't disable nametags
function on_nametags_render(index, name)
    if not ssHud then
        return ""
    end
end

hook_event(HOOK_ON_NAMETAGS_RENDER, on_nametags_render)
